/*
 *   libscrobblertag - A media file tag-reader library for AudioScrobbler
 *   Copyright (C) 2003  Pipian
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <musicbrainz/mb_c.h>
#include "endian.h"
#include "fmt.h"
#include "tags.h"
#include "unicode.h"
#define BUFFER_SIZE 4096

typedef struct {
	unsigned char *check;
	int count;
} resync;

static resync checkunsync(char *syncCheck)
{
	int i, j;
	resync sync;

	sync.check = syncCheck;
	sync.count = 0;

	for(i = 0; i < strlen(sync.check); i++)
	{
		if(sync.check[i] == 0xFF && sync.check[i + 1] == 0x00)
		{
			for(j = i + 1; j < strlen(sync.check) - 1; j++)
				syncCheck[j] = syncCheck[j + 1];
			sync.check[j] = '\0';
			sync.count++;
		}
	}

	return sync;
}

static resync checkunsync2(char *syncCheck, int size)
{
	int i, j;
	resync sync;

	sync.check = syncCheck;
	sync.count = 0;

	for(i = 0; i < size; i++)
	{
		if(sync.check[i] == 0xFF && sync.check[i + 1] == 0x00)
		{
			for(j = i + 1; j < size - 1; j++)
				sync.check[j] = sync.check[j + 1];
			sync.check[j] = '\0';
			sync.count++;
		}
	}

	return sync;
}

/* TODO: Implement SEEK tag */
static metadata id3v2_4(FILE *fp, int bottom)
{
	metadata meta;
	unsigned char *id3_frame_utf8 = NULL;
	char *tag_buffer, *bp = tag_buffer, cToInt[5] = "", id3_flags,
		*id3_frame_data = NULL, *id3_frame_data2, id3_frame_encoding;
	int id3_size, id3_frame_size, id3_unsync = 0, id3_extended = 0,
		id3_frame_compression = 0, id3_frame_encryption = 0,
		id3_frame_unsync = 0, id3_frame_grouping = 0,
		id3_frame_original_size = 0, id3_frame_size_decompressed = 0;

	clean_meta(&meta);

	pdebug("Version: ID3v2.4");

	fread(&id3_flags, 1, 1, fp);
	if((id3_flags & 0x80) == 0x80) id3_unsync = 1;
	if((id3_flags & 0x40) == 0x40) id3_extended = 1;
	fread(cToInt, 1, 4, fp);
	id3_size = synchsafe2int(cToInt);
	if(bottom == 1) fseek(fp, -10 - id3_size, SEEK_CUR);

	tag_buffer = malloc(id3_size);
	fread(tag_buffer, 1, id3_size, fp);
	bp = tag_buffer;

	if(id3_extended)
	{
		memcpy(cToInt, bp, 4);
		bp += 4;
		id3_frame_size = synchsafe2int(cToInt);
		bp += id3_frame_size;
	}
	while(bp - id3_size < tag_buffer)
	{
		id3_frame_compression = 0;
		id3_frame_encryption = 0;
		id3_frame_unsync = 0;
		id3_frame_grouping = 0;

		memcpy(cToInt, bp, 4);
		bp += 4;

		if(strcmp(cToInt, "TIT2") == 0)
		{
			pdebug("Found Title!");

			memcpy(cToInt, bp, 4);
			bp += 4;

			id3_frame_size = synchsafe2int(cToInt);
			bp++;
			id3_flags = *(bp++);
			if((id3_flags & 0x08) == 0x08) id3_frame_compression = 1;
			if((id3_flags & 0x04) == 0x04) id3_frame_encryption = 1;
			if(((id3_flags & 0x02) == 0x02) || id3_unsync)
				id3_frame_unsync = 1;
			if((id3_flags & 0x40) == 0x40) id3_frame_grouping = 1;
			if((id3_flags & 0x01) == 0x01) id3_frame_original_size = 1;
			if(id3_frame_compression || id3_frame_original_size)
			{
				memcpy(cToInt, bp, 4);
				bp += 4;
				id3_frame_size_decompressed = synchsafe2int(cToInt);
			}
			if(id3_frame_encryption) bp++;
			if(id3_frame_grouping) bp++;
			id3_frame_encoding = *(bp++);
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size);
			memset(id3_frame_data, '\0', id3_frame_size);
			memcpy(id3_frame_data, bp, id3_frame_size - 1);
			bp += id3_frame_size - 1;
			if(id3_frame_unsync) checkunsync(id3_frame_data);
			if(id3_frame_encoding == 0x00)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				iso88591_to_utf8(id3_frame_data, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x01)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16bom_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x02)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16be_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x03)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				id3_frame_utf8 = malloc(strlen(id3_frame_data) + 1);
				strcpy(id3_frame_utf8, id3_frame_data);
			}

			if(meta.title != NULL) free(meta.title);
			meta.title = malloc(strlen(id3_frame_utf8) + 1);
			strcpy(meta.title, id3_frame_utf8);
		}
		else if(strcmp(cToInt, "TPE1") == 0)
		{
			pdebug("Found Artist!");

			memcpy(cToInt, bp, 4);
			bp += 4;

			id3_frame_size = synchsafe2int(cToInt);
			bp++;
			id3_flags = *(bp++);
			if((id3_flags & 0x08) == 0x08) id3_frame_compression = 1;
			if((id3_flags & 0x04) == 0x04) id3_frame_encryption = 1;
			if(((id3_flags & 0x02) == 0x02) || id3_unsync)
				id3_frame_unsync = 1;
			if((id3_flags & 0x40) == 0x40) id3_frame_grouping = 1;
			if((id3_flags & 0x01) == 0x01) id3_frame_original_size = 1;
			if(id3_frame_compression || id3_frame_original_size)
			{
				memcpy(cToInt, bp, 4);
				bp += 4;
				id3_frame_size_decompressed = synchsafe2int(cToInt);
			}
			if(id3_frame_encryption) bp++;
			if(id3_frame_grouping) bp++;
			id3_frame_encoding = *(bp++);
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size);
			memset(id3_frame_data, '\0', id3_frame_size);
			memcpy(id3_frame_data, bp, id3_frame_size - 1);
			bp += id3_frame_size - 1;
			if(id3_frame_unsync) checkunsync(id3_frame_data);
			if(id3_frame_encoding == 0x00)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				iso88591_to_utf8(id3_frame_data, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x01)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16bom_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x02)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16be_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x03)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				id3_frame_utf8 = malloc(strlen(id3_frame_data) + 1);
				strcpy(id3_frame_utf8, id3_frame_data);
			}

			if(meta.artist != NULL) free(meta.artist);
			meta.artist = malloc(strlen(id3_frame_utf8) + 1);
			strcpy(meta.artist, id3_frame_utf8);
		}
		else if(strcmp(cToInt, "TALB") == 0)
		{
			pdebug("Found Album!");

			memcpy(cToInt, bp, 4);
			bp += 4;

			id3_frame_size = synchsafe2int(cToInt);
			bp++;
			id3_flags = *(bp++);
			if((id3_flags & 0x08) == 0x08) id3_frame_compression = 1;
			if((id3_flags & 0x04) == 0x04) id3_frame_encryption = 1;
			if(((id3_flags & 0x02) == 0x02) || id3_unsync)
				id3_frame_unsync = 1;
			if((id3_flags & 0x40) == 0x40) id3_frame_grouping = 1;
			if((id3_flags & 0x01) == 0x01) id3_frame_original_size = 1;
			if(id3_frame_compression || id3_frame_original_size)
			{
				memcpy(cToInt, bp, 4);
				bp += 4;
				id3_frame_size_decompressed = synchsafe2int(cToInt);
			}
			if(id3_frame_encryption) bp++;
			if(id3_frame_grouping) bp++;
			id3_frame_encoding = *(bp++);
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size);
			memset(id3_frame_data, '\0', id3_frame_size);
			memcpy(id3_frame_data, bp, id3_frame_size - 1);
			bp += id3_frame_size - 1;
			if(id3_frame_unsync) checkunsync(id3_frame_data);
			if(id3_frame_encoding == 0x00)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				iso88591_to_utf8(id3_frame_data, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x01)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16bom_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x02)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16be_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x03)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				id3_frame_utf8 = malloc(strlen(id3_frame_data) + 1);
				strcpy(id3_frame_utf8, id3_frame_data);
			}

			if(meta.album != NULL) free(meta.album);
			meta.album = malloc(strlen(id3_frame_utf8) + 1);
			strcpy(meta.album, id3_frame_utf8);
		}
		else if(strcmp(cToInt, "UFID") == 0)
		{
			int i;

			memcpy(cToInt, bp, 4);
			bp += 4;

			id3_frame_size = synchsafe2int(cToInt);
			bp++;
			id3_flags = *(bp++);
			if((id3_flags & 0x08) == 0x08) id3_frame_compression = 1;
			if((id3_flags & 0x04) == 0x04) id3_frame_encryption = 1;
			if(((id3_flags & 0x02) == 0x02) || id3_unsync)
				id3_frame_unsync = 1;
			if((id3_flags & 0x40) == 0x40) id3_frame_grouping = 1;
			if((id3_flags & 0x01) == 0x01) id3_frame_original_size = 1;
			if(id3_frame_compression || id3_frame_original_size)
			{
				memcpy(cToInt, bp, 4);
				bp += 4;
				id3_frame_size_decompressed = synchsafe2int(cToInt);
			}
			if(id3_frame_encryption) bp++;
			if(id3_frame_grouping) bp++;
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size + 1);
			memset(id3_frame_data, '\0', id3_frame_size + 1);
			memcpy(id3_frame_data, bp, id3_frame_size);
			bp += id3_frame_size;
			if(id3_frame_unsync) checkunsync2(id3_frame_data, id3_frame_size);
			if(strcmp(id3_frame_data, "http://musicbrainz.org") == 0)
			{
				pdebug("Found MusicBrainz Track ID!");

				id3_frame_data2 = id3_frame_data;
				for(i = 0; *id3_frame_data2 != '\0' &&
						i < id3_frame_size; i++)
							id3_frame_data2++;
				id3_frame_data2++;
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				id3_frame_utf8 = malloc(strlen(id3_frame_data2) + 1);
				strcpy(id3_frame_utf8, id3_frame_data2);

				if(meta.mb != NULL) free(meta.mb);
				meta.mb = malloc(strlen(id3_frame_utf8));
				strcpy(meta.mb, id3_frame_utf8);
			}
		}
		else if(cToInt[0] == '\0')
		{
			break;
		}
		else
		{
			memcpy(cToInt, bp, 4);
			bp += 4;

			id3_frame_size = synchsafe2int(cToInt);
			bp++;
			id3_flags = *(bp++);
			if((id3_flags & 0x08) == 0x08) id3_frame_compression = 1;
			if((id3_flags & 0x04) == 0x04) id3_frame_encryption = 1;
			if((id3_flags & 0x40) == 0x40) id3_frame_grouping = 1;
			if((id3_flags & 0x01) == 0x01) id3_frame_original_size = 1;
			if(id3_frame_compression || id3_frame_original_size) bp += 4;
			if(id3_frame_encryption) bp++;
			if(id3_frame_grouping) bp++;
			bp += id3_frame_size;
		}
	}

	free(tag_buffer);
	free(id3_frame_data);
	free(id3_frame_utf8);

	return meta;
} /* End ID3v2.4 */

static metadata id3v2_3(FILE *fp, int bottom)
{
	metadata meta;
	resync unsynced;
	unsigned char *id3_frame_utf8 = NULL;
	char *tag_buffer, *bp = tag_buffer, cToInt[5] = "", id3_flags,
		*syncFix = NULL, *id3_frame_data = NULL, *id3_frame_data2,
		id3_frame_encoding;
	int id3_size, id3_frame_size, id3_unsync = 0, id3_extended = 0,
		id3_frame_compression = 0, id3_frame_encryption = 0,
		id3_frame_unsync = 0, id3_frame_grouping = 0,
		i, id3_frame_size_decompressed;

	clean_meta(&meta);

	pdebug("Version: ID3v2.3");

	fread(&id3_flags, 1, 1, fp);
	if((id3_flags & 0x80) == 0x80) id3_unsync = 1;
	if((id3_flags & 0x40) == 0x40) id3_extended = 1;
	fread(cToInt, 1, 4, fp);
	id3_size = synchsafe2int(cToInt);
	if(bottom == 1) fseek(fp, -10 - id3_size, SEEK_CUR);

	tag_buffer = malloc(id3_size);
	fread(tag_buffer, 1, id3_size, fp);
	bp = tag_buffer;
	
	if(id3_extended)
	{
		memcpy(cToInt, bp, 4);
		bp += 4;
		if(id3_unsync == 1)
		{
			unsynced = checkunsync(cToInt);
			while(unsynced.count > 0)
			{
				if(syncFix != NULL) free(syncFix);
				syncFix = malloc(unsynced.count);
				memcpy(syncFix, bp, unsynced.count);
				bp += unsynced.count;
				for(i = 0; i < unsynced.count; i++)
					cToInt[4 - unsynced.count + i] = syncFix[i];
				unsynced = checkunsync(cToInt);
			}
			free(syncFix);
		}
		id3_frame_size = be2int(cToInt);
		bp += id3_frame_size;
	}
	while(bp - id3_size < tag_buffer)
	{
		id3_frame_compression = 0;
		id3_frame_encryption = 0;
		id3_frame_unsync = 0;
		id3_frame_grouping = 0;

		memcpy(cToInt, bp, 4);
		bp += 4;

		if(strcmp(cToInt, "TIT2") == 0)
		{
			pdebug("Found Title!");

			memcpy(cToInt, bp, 4);
			bp += 4;
			if(id3_unsync == 1)
			{
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
			}
			id3_frame_size = be2int(cToInt);
			bp++;
			id3_flags = *(bp++);
			if((id3_flags & 0x80) == 0x80) id3_frame_compression = 1;
			if((id3_flags & 0x40) == 0x40) id3_frame_encryption = 1;
			if(((id3_flags & 0x02) == 0x02) || id3_unsync) id3_frame_unsync = 1;
			if((id3_flags & 0x20) == 0x20) id3_frame_grouping = 1;
			if(id3_frame_compression && id3_frame_unsync)
			{
				memcpy(cToInt, bp, 4);
				bp += 4;
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				id3_frame_size_decompressed = be2int(cToInt);
			}
			if(id3_frame_encryption) bp++;
			if(id3_frame_grouping) bp++;
			id3_frame_encoding = *(bp++);
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size);
			memset(id3_frame_data, '\0', id3_frame_size);
			memcpy(id3_frame_data, bp, id3_frame_size - 1);
			bp += id3_frame_size - 1;
			if(id3_frame_unsync) checkunsync(id3_frame_data);
			if(id3_frame_encoding == 0x00)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				iso88591_to_utf8(id3_frame_data, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x01)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16bom_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}

			if(meta.title != NULL) free(meta.title);
			meta.title = malloc(strlen(id3_frame_utf8) + 1);
			strcpy(meta.title, id3_frame_utf8);
		}
		else if(strcmp(cToInt, "TPE1") == 0)
		{
			pdebug("Found Artist!");

			memcpy(cToInt, bp, 4);
			bp += 4;
			if(id3_unsync == 1)
			{
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
			}
			id3_frame_size = be2int(cToInt);
			bp++;
			id3_flags = *(bp++);
			if((id3_flags & 0x80) == 0x80) id3_frame_compression = 1;
			if((id3_flags & 0x40) == 0x40) id3_frame_encryption = 1;
			if(((id3_flags & 0x02) == 0x02) || id3_unsync) id3_frame_unsync = 1;
			if((id3_flags & 0x20) == 0x20) id3_frame_grouping = 1;
			if(id3_frame_compression && id3_frame_unsync)
			{
				memcpy(cToInt, bp, 4);
				bp += 4;
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				id3_frame_size_decompressed = be2int(cToInt);
			}
			if(id3_frame_encryption) bp++;
			if(id3_frame_grouping) bp++;
			id3_frame_encoding = *(bp++);
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size);
			memset(id3_frame_data, '\0', id3_frame_size);
			memcpy(id3_frame_data, bp, id3_frame_size - 1);
			bp += id3_frame_size - 1;
			if(id3_frame_unsync) checkunsync(id3_frame_data);
			if(id3_frame_encoding == 0x00)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				iso88591_to_utf8(id3_frame_data, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x01)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16bom_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}

			if(meta.artist != NULL) free(meta.artist);
			meta.artist = malloc(strlen(id3_frame_utf8) + 1);
			strcpy(meta.artist, id3_frame_utf8);
		}
		else if(strcmp(cToInt, "TALB") == 0)
		{
			pdebug("Found Album!");

			memcpy(cToInt, bp, 4);
			bp += 4;
			if(id3_unsync == 1)
			{
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
			}
			id3_frame_size = be2int(cToInt);
			bp++;
			id3_flags = *(bp++);
			if((id3_flags & 0x80) == 0x80) id3_frame_compression = 1;
			if((id3_flags & 0x40) == 0x40) id3_frame_encryption = 1;
			if(((id3_flags & 0x02) == 0x02) || id3_unsync) id3_frame_unsync = 1;
			if((id3_flags & 0x20) == 0x20) id3_frame_grouping = 1;
			if(id3_frame_compression && id3_frame_unsync)
			{
				memcpy(cToInt, bp, 4);
				bp += 4;
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
				id3_frame_size_decompressed = be2int(cToInt);
			}
			if(id3_frame_encryption) bp++;
			if(id3_frame_grouping) bp++;
			id3_frame_encoding = *(bp++);
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size);
			memset(id3_frame_data, '\0', id3_frame_size);
			memcpy(id3_frame_data, bp, id3_frame_size - 1);
			bp += id3_frame_size - 1;
			if(id3_frame_unsync) checkunsync(id3_frame_data);
			if(id3_frame_encoding == 0x00)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				iso88591_to_utf8(id3_frame_data, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x01)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16bom_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}

			if(meta.album != NULL) free(meta.album);
			meta.album = malloc(strlen(id3_frame_utf8) + 1);
			strcpy(meta.album, id3_frame_utf8);
		}
		else if(strcmp(cToInt, "UFID") == 0)
		{
			memcpy(cToInt, bp, 4);
			bp += 4;
			if(id3_unsync == 1)
			{
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
			}
			id3_frame_size = be2int(cToInt);
			bp++;
			id3_flags = *(bp++);
			if((id3_flags & 0x80) == 0x80) id3_frame_compression = 1;
			if((id3_flags & 0x40) == 0x40) id3_frame_encryption = 1;
			if(((id3_flags & 0x02) == 0x02) || id3_unsync) id3_frame_unsync = 1;
			if((id3_flags & 0x20) == 0x20) id3_frame_grouping = 1;
			if(id3_frame_compression && id3_frame_unsync)
			{
				memcpy(cToInt, bp, 4);
				bp += 4;
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
				id3_frame_size_decompressed = be2int(cToInt);
			}
			if(id3_frame_encryption) bp++;
			if(id3_frame_grouping) bp++;
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size + 1);
			memset(id3_frame_data, '\0', id3_frame_size + 1);
			memcpy(id3_frame_data, bp, id3_frame_size);
			bp += id3_frame_size;
			if(id3_frame_unsync) checkunsync2(id3_frame_data, id3_frame_size);
			if(strcmp(id3_frame_data, "http://musicbrainz.org") == 0)
			{
				pdebug("Found MusicBrainz Track ID!");

				id3_frame_data2 = id3_frame_data;
				for(i = 0; *id3_frame_data2 != '\0' &&
						i < id3_frame_size; i++)
							id3_frame_data2++;
				id3_frame_data2++;
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				id3_frame_utf8 = malloc(strlen(id3_frame_data2) + 1);
				strcpy(id3_frame_utf8, id3_frame_data2);

				if(meta.mb != NULL) free(meta.mb);
				meta.mb = malloc(strlen(id3_frame_utf8) + 1);
				strcpy(meta.mb, id3_frame_utf8);
			}
		}
		else if(cToInt[0] == '\0')
		{
			break;
		}
		else
		{
			memcpy(cToInt, bp, 4);
			bp += 4;

			if(id3_unsync == 1)
			{
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
			}
			id3_frame_size = be2int(cToInt);
			bp++;
			id3_flags = *(bp++);
			if((id3_flags & 0x80) == 0x80) id3_frame_compression = 1;
			if((id3_flags & 0x40) == 0x40) id3_frame_encryption = 1;
			if(((id3_flags & 0x02) == 0x02) || id3_unsync) id3_frame_unsync = 1;
			if((id3_flags & 0x20) == 0x20) id3_frame_grouping = 1;
			if(id3_frame_compression && id3_frame_unsync)
			{
				memcpy(cToInt, bp, 4);
				bp += 4;
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
				id3_frame_size_decompressed = be2int(cToInt);
			}
			if(id3_frame_encryption) bp++;
			if(id3_frame_grouping) bp++;
			bp += id3_frame_size;
		}
	}

	free(tag_buffer);
	free(id3_frame_data);
	free(id3_frame_utf8);

	return meta;
} /* End ID3v2.3 */

static metadata id3v2_2(FILE *fp)
{
	metadata meta;
	unsigned char *id3_frame_utf8 = NULL;
	char *tag_buffer, *bp = tag_buffer, cToInt[4] = "", id3_flags,
		*syncFix = NULL, *id3_frame_data = NULL, *id3_frame_data2,
		id3_frame_encoding;
	int id3_size, id3_frame_size, id3_unsync = 0, id3_frame_compression = 0,
		id3_frame_encryption = 0, id3_frame_unsync = 0, id3_frame_grouping = 0,
		id3_compression = 0, i;
	resync unsynced;

	clean_meta(&meta);

	pdebug("Version: ID3v2.2");

	fread(&id3_flags, 1, 1, fp);
	if((id3_flags & 0x80) == 0x80) id3_unsync = 1;
	if((id3_flags & 0x40) == 0x40) id3_compression = 1;
	fread(cToInt, 1, 4, fp);
	id3_size = synchsafe2int(cToInt);

	tag_buffer = malloc(id3_size);
	fread(tag_buffer, 1, id3_size, fp);
	bp = tag_buffer;

	while(bp - id3_size < tag_buffer)
	{
		id3_frame_compression = 0;
		id3_frame_encryption = 0;
		id3_frame_unsync = 0;
		id3_frame_grouping = 0;

		memcpy(cToInt, bp, 3);
		cToInt[3] = '\0';
		bp += 3;
		if(strcmp(cToInt, "TT2") == 0)
		{
			pdebug("Found Title!");

			memcpy(cToInt, bp, 3);
			bp += 3;
			if(id3_unsync == 1)
			{
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
			}
			id3_frame_size = be24int(cToInt);
			id3_frame_encoding = *(bp++);
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size);
			memset(id3_frame_data, '\0', id3_frame_size);
			memcpy(id3_frame_data, bp, id3_frame_size - 1);
			bp += id3_frame_size - 1;
			if(id3_frame_unsync) checkunsync(id3_frame_data);
			if(id3_frame_encoding == 0x00)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				iso88591_to_utf8(id3_frame_data, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x01)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16bom_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}

			if(meta.title != NULL) free(meta.title);
			meta.title = malloc(strlen(id3_frame_utf8) + 1);
			strcpy(meta.title, id3_frame_utf8);
		}
		else if(strcmp(cToInt, "TP1") == 0)
		{
			pdebug("Found Artist!");

			memcpy(cToInt, bp, 3);
			bp += 3;
			if(id3_unsync == 1)
			{
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
			}
			id3_frame_size = be24int(cToInt);
			id3_frame_encoding = *(bp++);
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size);
			memset(id3_frame_data, '\0', id3_frame_size);
			memcpy(id3_frame_data, bp, id3_frame_size - 1);
			bp += id3_frame_size - 1;
			if(id3_frame_unsync) checkunsync(id3_frame_data);
			if(id3_frame_encoding == 0x00)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				iso88591_to_utf8(id3_frame_data, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x01)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16bom_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}

			if(meta.artist != NULL) free(meta.artist);
			meta.artist = malloc(strlen(id3_frame_utf8) + 1);
			strcpy(meta.artist, id3_frame_utf8);
		}
		else if(strcmp(cToInt, "TAL") == 0)
		{
			pdebug("Found Album!");

			memcpy(cToInt, bp, 3);
			bp += 3;
			if(id3_unsync == 1)
			{
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
			}
			id3_frame_size = be24int(cToInt);
			id3_frame_encoding = *(bp++);
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size);
			memset(id3_frame_data, '\0', id3_frame_size);
			memcpy(id3_frame_data, bp, id3_frame_size - 1);
			bp += id3_frame_size - 1;
			if(id3_frame_unsync) checkunsync(id3_frame_data);
			if(id3_frame_encoding == 0x00)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				iso88591_to_utf8(id3_frame_data, &id3_frame_utf8);
			}
			else if(id3_frame_encoding == 0x01)
			{
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				utf16bom_to_utf8(id3_frame_data, id3_frame_size - 1, &id3_frame_utf8);
			}

			if(meta.album != NULL) free(meta.album);
			meta.album = malloc(strlen(id3_frame_utf8) + 1);
			strcpy(meta.album, id3_frame_utf8);
		}
		else if(strcmp(cToInt, "UFI") == 0)
		{
			memcpy(cToInt, bp, 3);
			bp += 3;
			if(id3_unsync == 1)
			{
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
			}
			id3_frame_size = be24int(cToInt);
			id3_frame_encoding = *(bp++);
			if(id3_frame_data != NULL) free(id3_frame_data);
			id3_frame_data = malloc(id3_frame_size + 1);
			memset(id3_frame_data, '\0', id3_frame_size + 1);
			memcpy(id3_frame_data, bp, id3_frame_size);
			bp += id3_frame_size;
			if(id3_frame_unsync) checkunsync2(id3_frame_data, id3_frame_size);
			if(strcmp(id3_frame_data, "http://musicbrainz.org") == 0)
			{
				pdebug("Found MusicBrainz Track ID!");

				id3_frame_data2 = id3_frame_data;
				for(i = 0; *id3_frame_data2 != '\0' &&
						i < id3_frame_size; i++)
							id3_frame_data2++;
				id3_frame_data2++;
				if(id3_frame_utf8 != NULL) free(id3_frame_utf8);
				id3_frame_utf8 = malloc(strlen(id3_frame_data2) + 1);
				strcpy(id3_frame_utf8, id3_frame_data2);

				if(meta.mb != NULL) free(meta.mb);
				meta.mb = malloc(strlen(id3_frame_utf8) + 1);
				strcpy(meta.mb, id3_frame_utf8);
			}
		}
		else if(cToInt[0] == '\0')
		{
			break;
		}
		else
		{
			memcpy(cToInt, bp, 3);
			bp += 3;
			if(id3_unsync == 1)
			{
				unsynced = checkunsync(cToInt);
				while(unsynced.count > 0)
				{
					if(syncFix != NULL) free(syncFix);
					syncFix = malloc(unsynced.count);
					memcpy(syncFix, bp, unsynced.count);
					bp += unsynced.count;
					for(i = 0; i < unsynced.count; i++)
						cToInt[4 - unsynced.count + i] = syncFix[i];
					unsynced = checkunsync(cToInt);
				}
				free(syncFix);
				syncFix = NULL;
			}
			id3_frame_size = be24int(cToInt);
			bp += id3_frame_size;
		}
	}

	free(tag_buffer);
	free(id3_frame_data);
	free(id3_frame_utf8);

	return meta;
} /* End ID3v2.2 */

char *tag_exists(char *filename)
{
	FILE *fp;
	int search = -1, i, status = 0, charsRead;
	long pos = 0;
	char *tag_id;
	unsigned char tag_buffer[BUFFER_SIZE], *bp = tag_buffer, vorbis_type;
	static char tags[2];

	tags[0] = NO_TAGS;
	tags[1] = NO_TAGS;

	/*
	 * Check for CD Audio
	 * There's probably a better way than just checking the filename
	 * for .cda at the end...  But I bet it's OS specific.
	 *
	 * TODO: Make and use strcasecmp in fmt.c
	 *
	 */
	if(fmt_strcasecmp(strrchr(filename, '.') + 1, "cda") == 0)
	{
		pdebug("Found CD Audio...");
		tags[0] = CD_AUDIO;

		return tags;
	}

	fp = fopen(filename, "r");
	if(fp == NULL)
	{
		pdebug("Error reading file...");
		tags[0] = READ_ERROR;

		return tags;
	}

	/* Check for ID3v1 */
	fseek(fp, -128, SEEK_END);
	tag_id = malloc(4);
	memset(tag_id, '\0', 4);
	charsRead = fread(tag_id, 1, 3, fp);
	if(strcmp(tag_id, "TAG") == 0)
	{
		pdebug("Found ID3v1 tag...");
		tags[1] |= ID3V1_TAG;
	}

	/* Check for ID3v2 */
	fseek(fp, 0, SEEK_SET);
	charsRead = fread(tag_buffer, 1, 10, fp);
	bp = tag_buffer;
	while(status == 0)
	{
		if(search == -1)
		{
			if((strncmp(bp, "ID3", 3) == 0 ||
				strncmp(bp, "3DI", 3) == 0)) status = 1;
			fseek(fp, 3, SEEK_END);
			charsRead = fread(tag_buffer, 1, 3, fp);
			search = -2;
		}
		else
		{
			if(search == -2)
			{
				bp = tag_buffer;
				if((strncmp(bp, "ID3", 3) == 0 ||
					strncmp(bp, "3DI", 3) == 0)) status = 1;
				pos = ftell(fp) - BUFFER_SIZE;
				search = 1;
			}
			fseek(fp, pos, SEEK_SET);
			charsRead = fread(tag_buffer, 1, BUFFER_SIZE, fp);

			bp = tag_buffer;
			for(i = 0; i < charsRead - 3 && status == 0; i++)
			{
				bp++;
				if((strncmp(bp, "ID3", 3) == 0 ||
					strncmp(bp, "3DI", 3) == 0)) status = 1;
			}
			pos -= BUFFER_SIZE - 9;
			if((pos < (-BUFFER_SIZE + 9) || ferror(fp)) &&
					status != 1)
						status = -1;
		}
		if(status == 1 && *(bp + 3) < 0xFF && *(bp + 4) < 0xFF &&
				*(bp + 6) < 0x80 && *(bp + 7) < 0x80 && *(bp + 8) < 0x80 &&
				*(bp + 9) < 0x80)
					status = 1;
		else if(status != -1) status = 0;
		if(search == 0) search = -1;
	}
	if(status == 1)
	{
		pdebug("Found ID3v2 tag...");
		tags[1] |= ID3V2_TAG;
	}
	status = 0;

	/* Check for Ogg Vorbis */
	fseek(fp, 0, SEEK_SET);
	realloc(tag_id, 5);
	memset(tag_id, '\0', 5);
	fread(tag_id, 1, 4, fp);
	if(strcmp(tag_id, "OggS") == 0)
	{
		fread(tag_buffer, 1, BUFFER_SIZE, fp);
		bp = tag_buffer;
		while(status == 0)
		{
			for(i = 0; i < BUFFER_SIZE - 6 && status == 0; i++)
			{
				if(strncmp(bp, "vorbis", 6) == 0)
				{
					vorbis_type = *(bp - 1);
					if(vorbis_type == 0x03) status = 1;
				}
				bp++;
			}
			if(status == 1 || feof(fp)) break;
			memmove(tag_buffer, tag_buffer + BUFFER_SIZE - 5, 5);
			fread(tag_buffer + 5, 1, BUFFER_SIZE - 5, fp);
			bp = tag_buffer;
		}
		if(status == 1)
		{
			pdebug("Found Vorbis comment block...");
			tags[1] |= VORBIS_TAG;
		}
	}
	status = 0;

	/* Check for FLAC */
	fseek(fp, 0, SEEK_SET);
	fread(tag_id, 1, 4, fp);
	if(strcmp(tag_id, "fLaC") == 0)
	{
		fread(tag_id, 1, 4, fp);
		while(!feof(fp))
		{
			if((tag_id[0] & 0x7F) == 0x04) break;
			if((tag_id[0] & 0x80) == 0x80) break;
			pos = flac2int(tag_id);
			fseek(fp, pos, SEEK_CUR);
			fread(tag_id, 1, 4, fp);
		}
		if((tag_id[0] & 0x7F) == 0x04)
		{
			pdebug("Found FLAC tag...");
			tags[1] |= FLAC_TAG;
		}
	}

	/* Check for APE */
	fseek(fp, 0, SEEK_SET);
	charsRead = fread(tag_buffer, 1, BUFFER_SIZE, fp);
	bp = tag_buffer;
	while(status == 0)
	{
		for(i = 0; i < BUFFER_SIZE - 8 && status == 0; i++)
		{
			bp++;
			if(strncmp(bp, "APETAGEX", 8) == 0) status = 1;
		}
		if(status == 1 || feof(fp)) break;
		memmove(tag_buffer, tag_buffer + BUFFER_SIZE - 7, 7);
		charsRead = fread(tag_buffer + 7, 1, BUFFER_SIZE - 7, fp);
		bp = tag_buffer;
	}
	if(status == 1)
	{
		pdebug("Found APE tag...");
		tags[1] |= APE_TAG;
	}
	status = 0;

	free(tag_id);

	fclose(fp);

	return tags;
}

static metadata metaCD(char *filename, char track)
{
	int retVal;
	metadata meta;
	musicbrainz_t mb;
	char *tmp;

	clean_meta(&meta);

	/*
	 * A lot of this is going to be system-specific,
	 * so you'd best change the includes for your system
	 * and write your own version of the getToC and submitRDF functions.
	 * The rest is right here in this file.
	 *
	 * Or now that we've switched to MusicBrainz...
	 *
	 */
	tmp = malloc(BUFFER_SIZE / 4 + 1);
	mb = mb_New();
	mb_SetDevice(mb, filename);
	pdebug("Getting CD Audio metadata...");
	pdebug("Submitting query to MusicBrainz...");
	retVal = mb_Query(mb, MBQ_GetCDInfo);
	if(retVal == 0)
	{
		char error[129] = "";
		pdebug("ERROR: Query failed.");
		mb_GetQueryError(mb, error, 128);
		pdebug(fmt_vastr("REASON: %s", error));
		mb_Delete(mb);
		free(tmp);
		return meta;
	}
	pdebug("Selecting result...");
	retVal = mb_Select1(mb, MBS_SelectAlbum, 1);
	if(retVal == 0)
	{
		pdebug("ERROR: Album select failed.");
		mb_Delete(mb);
		free(tmp);
		return meta;
	}
	pdebug("Extracting metadata from result...");
	memset(tmp, '\0', BUFFER_SIZE / 4 + 1);
	retVal = mb_GetResultData(mb, MBE_AlbumGetAlbumName, tmp, BUFFER_SIZE / 4);
	if(retVal == 0)
	{
		pdebug("ERROR: Album title not found.");
		meta.album = "";
	}
	else
	{
		meta.album = malloc(strlen(tmp) + 1);
		memset(meta.album, '\0', strlen(tmp) + 1);
		strcpy(meta.album, tmp);
	}
	memset(tmp, '\0', BUFFER_SIZE / 4 + 1);
	retVal = mb_GetResultData1(mb, MBE_AlbumGetArtistName, tmp, BUFFER_SIZE / 4, track);
	if(retVal == 0)
	{
		pdebug("ERROR: Artist name not found.");
		meta.artist = "";
	}
	else
	{
		meta.artist = malloc(strlen(tmp) + 1);
		memset(meta.artist, '\0', strlen(tmp) + 1);
		strcpy(meta.artist, tmp);
	}
	memset(tmp, '\0', BUFFER_SIZE / 4 + 1);
	retVal = mb_GetResultData1(mb, MBE_AlbumGetTrackName, tmp, BUFFER_SIZE / 4, track);
	if(retVal == 0)
	{
		pdebug("ERROR: Track title not found.");
		meta.title = "";
	}
	else
	{
		meta.title = malloc(strlen(tmp) + 1);
		memset(meta.title, '\0', strlen(tmp) + 1);
		strcpy(meta.title, tmp);
	}
	memset(tmp, '\0', BUFFER_SIZE / 4 + 1);
	retVal = mb_GetResultData1(mb, MBE_AlbumGetTrackId, tmp, 36, track);
	if(retVal == 0)
	{
		pdebug("ERROR: MBID not found.");
		meta.mb = "";
	}
	else
	{
		meta.mb = malloc(37);
		memset(meta.mb, '\0', 37);
		strcpy(meta.mb, tmp);
	}
	mb_Delete(mb);
	free(tmp);

	return meta;
} /* End CD Audio support */

static metadata metaAPE(char *filename)
{
	metadata meta;
	FILE *fp;
	unsigned char ape_item_key[256 * 6] = "",
			*ape_item_value = NULL, *ape_item_subvalue;
	unsigned char cToInt[4], *tag_buffer, *bp,
		*uBuffer = NULL, ape_flags[4];
	int pos = 0, ape_items, ape_size, ape_start, ape_version, try = 0, i, j,
			status = 0, pb = 0;

	fp = fopen(filename, "rb");

	clean_meta(&meta);

	pdebug("Getting APE tag metadata...");
	pdebug("Searching for tag...");
	/* Find the tag header or footer and point the file pointer there. */
	fseek(fp, 0, SEEK_SET);
	tag_buffer = malloc(BUFFER_SIZE);
	pb += fread(tag_buffer, 1, BUFFER_SIZE, fp);
	bp = tag_buffer;
	while(status == 0)
	{
		for(i = 0; i < BUFFER_SIZE - 8 && status == 0; i++)
		{
			bp++;
			if(strncmp(bp, "APETAGEX", 8) == 0) status = 1;
		}
		if(status == 1 || feof(fp)) break;
		memmove(tag_buffer, tag_buffer + BUFFER_SIZE - 7, 7);
		pos += pb - pos;
		pb += fread(tag_buffer + 7, 1, BUFFER_SIZE - 7, fp);
		bp = tag_buffer;
	}
	fseek(fp, pos + (bp - tag_buffer) + 1, SEEK_SET);

	/* And then get the tag version. */
	fread(cToInt, 1, 4, fp);
	ape_version = le2int(cToInt);
	pdebug(ape_version == 1000 ? "Found APE1 tag..." :
			ape_version == 2000 ? "Found APE2 tag..." :
			"Found unknown APE tag...");

	/*
	 * Now for the actual reading.
	 * APE2 and APE1 tags are identical for all the structure we care about.
	 *
	 */

	fread(cToInt, 1, 4, fp);
	ape_size = le2int(cToInt);
	fread(cToInt, 1, 4, fp);
	ape_items = le2int(cToInt);
	pdebug(fmt_vastr("tag size: %d", ape_size));
	pdebug(fmt_vastr("items: %d", ape_items));
	fread(ape_flags, 1, 4, fp);
	/* Is it a footer?  Header? */
	if((ape_flags[3] & 0x20) == 0x0 || ape_version == 1000)
		ape_start = -24 - ape_size;
	else ape_start = 8;
	fseek(fp, ape_start, SEEK_CUR);

	realloc(tag_buffer, ape_size);
	fread(tag_buffer, 1, ape_size, fp);
	bp = tag_buffer;

	pdebug("Searching APE tag for Title, Artist, Album, and MusicBrainz...");
	for(i = 1; i < ape_items; i++)
	{
		char ch;

		memcpy(cToInt, bp, 4);
		ape_size = le2int(cToInt);
		bp += 8;
		ch = *(bp++);
		for(j = 0; j < 255 * 6 && ch != '\0'; j++)
		{
			ape_item_key[j] = ch;
			ch = *(bp++);
		}
		if(uBuffer != NULL) free(uBuffer);
		uBuffer = malloc(ape_size + 1);
		memset(uBuffer, '\0', ape_size + 1);
		if(ape_item_value != NULL) free(ape_item_value);
		ape_item_value = malloc(ape_size + 1);
		memset(ape_item_value, '\0', ape_size + 1);
		memcpy(uBuffer, bp, ape_size);
		bp += ape_size;
		strcpy(uBuffer, ape_item_value);

		if(strcmp(ape_item_key, "Title") == 0)
		{
			pdebug("Found Title!");
			if(meta.title != NULL) free(meta.title);
			meta.title = malloc(strlen(ape_item_value) + 1);
			strcpy(meta.title, ape_item_value);
		}
		else if(strcmp(ape_item_key, "Artist") == 0)
		{
			pdebug("Found Artist!");
			if(meta.artist != NULL) free(meta.artist);
			meta.artist = malloc(strlen(ape_item_value) + 1);
			strcpy(meta.artist, ape_item_value);
		}
		else if(strcmp(ape_item_key, "Album") == 0)
		{
			pdebug("Found Album!");
			if(meta.album != NULL) free(meta.album);
			meta.album = malloc(strlen(ape_item_value) + 1);
			strcpy(meta.album, ape_item_value);
		}
		else if(strcmp(ape_item_key, "Comment") == 0)
		{
			unsigned char *comment_value = NULL, *eq_pos, *sep_pos;

			ape_item_subvalue = ape_item_value;
			sep_pos = strchr(ape_item_subvalue, '|');
			while(sep_pos != NULL || try == 0)
			{
				try = 1;
				if(sep_pos != NULL) *sep_pos = '\0';
				if(comment_value != NULL) free(comment_value);
				comment_value = malloc(strlen(ape_item_subvalue) + 1);
				strcpy(comment_value, ape_item_subvalue);
				if(sep_pos != NULL) sep_pos++;
				eq_pos = strchr(comment_value, '=');
				if(eq_pos != NULL)
				{
					*eq_pos = '\0';
					eq_pos++;
					if(strcmp(comment_value, "musicbrainz_trackid") == 0)
					{
						pdebug("Found MusicBrainz Track ID!");
						if(meta.mb != NULL) free(meta.mb);
						meta.mb = malloc(strlen(eq_pos) + 1);
						strcpy(meta.mb, eq_pos);
						break;
					}
				}
				if(sep_pos != NULL)
				{
					try = 0;
					ape_item_subvalue = sep_pos;
					sep_pos = strchr(ape_item_subvalue, '|');
				}
			}
			try = 0;
			free(comment_value);
		}
		memset(ape_item_key, '\0', 256 * 6);
	}

	fclose(fp);

	free(tag_buffer);
	free(uBuffer);
	free(ape_item_value);

	meta.tag_type[1] = APE_TAG;

	return meta;
} /* End APE support */

static metadata metaVorbis(char *filename, char tag_type[2])
{
	metadata meta;
	FILE *fp;
	unsigned char *vorbis_item = NULL;
	unsigned char *tag_buffer, *bp, vorbis_type = 0x00, cToInt[4],
		*uBuffer = NULL;
	int pos = 4, len, vorbis_items, i, status = 0;

	fp = fopen(filename, "rb");

	clean_meta(&meta);

	fseek(fp, 0, SEEK_SET);

	/* Two slightly different methods of getting to the data... */
	/* First comes Vorbis... */
	if((tag_type[1] & VORBIS_TAG) == VORBIS_TAG)
	{
		pdebug("Getting Vorbis comment metadata...");
		pdebug("Searching for tag...");
		tag_buffer = malloc(BUFFER_SIZE);
		fseek(fp, 4, SEEK_CUR);
		fread(tag_buffer, 1, BUFFER_SIZE, fp);
		bp = tag_buffer;
		while(status == 0)
		{
			for(i = 0; i < BUFFER_SIZE - 6 && status == 0; i++)
			{
				if(strncmp(bp, "vorbis", 6) == 0)
				{
					vorbis_type = *(bp - 1);
					if(vorbis_type == 0x03)
					{
						pos = bp - tag_buffer + pos;
						status = 1;
					}
				}
				bp++;
			}
			if(status == 1 || feof(fp)) break;
			memmove(tag_buffer, tag_buffer + BUFFER_SIZE - 5, 5);
			pos += fread(tag_buffer + 5, 1, BUFFER_SIZE - 5, fp);
			bp = tag_buffer;
		}
		free(tag_buffer);
	}
	/* Then comes FLAC */
	else if((tag_type[1] & FLAC_TAG) == FLAC_TAG)
	{
		pdebug("Getting FLAC tag metadata...");
		pdebug("Searching for tag...");
		uBuffer = malloc(4);
		fseek(fp, 4, SEEK_CUR);
		fread(uBuffer, 1, 4, fp);
		while(!feof(fp))
		{
			if((uBuffer[0] & 0x7F) == 0x04)
			{
				pdebug("Found FLAC tag...");
				break;
			}
			if((uBuffer[0] & 0x80) == 0x80) break;
			pos = flac2int(uBuffer);
			fseek(fp, pos, SEEK_CUR);
			fread(uBuffer, 1, 4, fp);
		}
	}
	/* But the internal data structure is the same. */
	if(vorbis_type == 0x03 || (uBuffer[0] & 0x7F) == 0x04)
	{
		if(vorbis_type == 0x03) fseek(fp, pos + 6, SEEK_SET);

		fread(cToInt, 1, 4, fp);
		fseek(fp, le2int(cToInt), SEEK_CUR);
		fread(cToInt, 1, 4, fp);
		vorbis_items = le2int(cToInt);
		pdebug(fmt_vastr("items: %d", vorbis_items));

		pdebug("Searching tag for Title, Artist, Album, and MusicBrainz...");

		/* TODO: Fix capitalization problems with a strcasecmp */
		for(i = 0; i < vorbis_items; i++)
		{
			char *eq_pos;

			fread(cToInt, 1, 4, fp);
			len = le2int(cToInt);
			if(uBuffer != NULL) free(uBuffer);
			uBuffer = malloc(len + 1);
			memset(uBuffer, '\0', len + 1);
			if(vorbis_item != NULL) free(vorbis_item);
			vorbis_item = malloc(len + 1);
			memset(vorbis_item, '\0', len + 1);
			fread(uBuffer, 1, len, fp);
			strcpy(vorbis_item, uBuffer);

			eq_pos = strchr(vorbis_item, '=');
			if(eq_pos != NULL)
			{
				*eq_pos = '\0';
				eq_pos++;
				if(fmt_strcasecmp(vorbis_item, "TITLE") == 0)
				{
					pdebug("Found Title!");
					if(meta.title != NULL) free(meta.title);
					meta.title = malloc(strlen(eq_pos) + 1);
					strcpy(meta.title, eq_pos);
				}
				else if(fmt_strcasecmp(vorbis_item, "ARTIST") == 0)
				{
					pdebug("Found Artist!");
					if(meta.artist != NULL) free(meta.artist);
					meta.artist = malloc(strlen(eq_pos) + 1);
					strcpy(meta.artist, eq_pos);
				}
				else if(fmt_strcasecmp(vorbis_item, "ALBUM") == 0)
				{
					pdebug("Found Album!");
					if(meta.album != NULL) free(meta.album);
					meta.album = malloc(strlen(eq_pos) + 1);
					strcpy(meta.album, eq_pos);
				}
				else if(fmt_strcasecmp(vorbis_item,
						"MUSICBRAINZ_TRACKID") == 0)
				{
					pdebug("Found MusicBrainz Track ID!");
					if(meta.mb != NULL) free(meta.mb);
					meta.mb = malloc(strlen(eq_pos) + 1);
					strcpy(meta.mb, eq_pos);
				}
			}
		}
	}

	fclose(fp);

	free(uBuffer);
	free(vorbis_item);

	if(vorbis_type == 0x03) meta.tag_type[1] = VORBIS_TAG;
	else meta.tag_type[1] = FLAC_TAG;

	return meta;
} /* End Vorbis/FLAC support */

static metadata metaID3v2(char *filename)
{
	FILE *fp;
	metadata meta;
	unsigned char tag_buffer[BUFFER_SIZE], *bp = tag_buffer;
	char id3_version[2];
	int pos, search = -1, i, bottom = 0, status = 0, charsRead;

	fp = fopen(filename, "rb");

	clean_meta(&meta);

	fseek(fp, 0, SEEK_SET);

	charsRead = fread(tag_buffer, 1, 10, fp);
	pos = 3;
	bp = tag_buffer;
	pdebug("Getting ID3v2 tag metadata...");
	pdebug("Searching for tag...");
	while(status == 0)
	{
		if(search == -1)
		{
			if((strncmp(bp, "ID3", 3) == 0 ||
				strncmp(bp, "3DI", 3) == 0)) status = 1;
			fseek(fp, 3, SEEK_END);
			charsRead = fread(tag_buffer, 1, 3, fp);
			search = -2;
		}
		else
		{
			if(search == -2)
			{
				bp = tag_buffer;
				pos = ftell(fp) - 3;
				if((strncmp(bp, "ID3", 3) == 0 ||
					strncmp(bp, "3DI", 3) == 0)) status = 1;
				search = 1;
			}
			if(status != 1)
			{
				pos = ftell(fp) - BUFFER_SIZE;
				fseek(fp, pos, SEEK_SET);
				charsRead = fread(tag_buffer, 1, BUFFER_SIZE, fp);

				bp = tag_buffer;
				for(i = 0; i < charsRead - 3 && status == 0; i++)
				{
					bp++;
					if((strncmp(bp, "ID3", 3) == 0 ||
						strncmp(bp, "3DI", 3) == 0)) status = 1;
				}
				if(status == 1) pos += bp - tag_buffer;
				pos -= BUFFER_SIZE - 9;
				if((pos < -BUFFER_SIZE + 9 || ferror(fp)) &&
						status != 1)
							status = -1;
			}
		}
		if(status == 1 && *(bp + 3) < 0xFF && *(bp + 4) < 0xFF &&
				*(bp + 6) < 0x80 && *(bp + 7) < 0x80 && *(bp + 8) < 0x80 &&
				*(bp + 9) < 0x80)
					status = 1;
		else if(status != -1) status = 0;
		if(search == 0) search = -1;
	}
	if(status == 1)
	{
		fseek(fp, pos, SEEK_SET);
		if(strncmp(bp, "3DI", 3) == 0) bottom = 1;
		pdebug("Found ID3v2 tag...");
		fread(id3_version, 1, 2, fp);
		/*
		 * ID3v2.3 and ID3v2.4 are close enough,
		 * even if the former is nastier...
		 * They still have separate functions though.
		 *
		 */
		if(id3_version[0] == 0x04) meta = id3v2_4(fp, bottom);
		else if(id3_version[0] == 0x03) meta = id3v2_3(fp, bottom);
		/* Now for that obsolete friend, ID3v2.2.  At least it's shorter. */
		else if(id3_version[0] == 0x02) meta = id3v2_2(fp);
	}

	fclose(fp);

	meta.tag_type[1] = ID3V2_TAG;

	return meta;
} /* End ID3v2 support */

static void cleanID3v1(unsigned char *meta)
{
	int i;
	
	for(i = strlen(meta) - 1; i > -1; i--)
	{
		if(meta[i] == ' ') meta[i] = '\0';
		else break;
	}
}

static metadata metaID3v1(char *filename)
{
	metadata meta;
	FILE *fp;
	char *uBuffer;
	
	fp = fopen(filename, "rb");

	clean_meta(&meta);

	fseek(fp, -128, SEEK_END);
	fseek(fp, 3, SEEK_CUR);
	uBuffer = malloc(31);
	memset(uBuffer, '\0', 31);
	fread(uBuffer, 1, 30, fp);
	cleanID3v1(uBuffer);
	if(meta.title != NULL) free(meta.title);
	iso88591_to_utf8(uBuffer, &meta.title);
	fread(uBuffer, 1, 30, fp);
	cleanID3v1(uBuffer);
	if(meta.artist != NULL) free(meta.artist);
	iso88591_to_utf8(uBuffer, &meta.artist);
	fread(uBuffer, 1, 30, fp);
	cleanID3v1(uBuffer);
	if(meta.album != NULL) free(meta.album);
	iso88591_to_utf8(uBuffer, &meta.artist);

	fclose(fp);

	free(uBuffer);

	meta.tag_type[1] = ID3V1_TAG;

	return meta;
} /* End ID3v1 support */

metadata get_tag_data(char *filename, char tag_type[2])
{
	metadata meta;

	clean_meta(&meta);

	/*
	 * Getting CD Audio metadata from MusicBrainz...
	 *
	 */
	if((tag_type[0] & CD_AUDIO) == CD_AUDIO)
		meta = metaCD(filename, tag_type[1]);
	/* Getting APE metadata */
	else if((tag_type[1] & APE_TAG) == APE_TAG) meta = metaAPE(filename);
	/* Getting Vorbis and FLAC metadata */
	else if((tag_type[1] & VORBIS_TAG) == VORBIS_TAG ||
			(tag_type[1] & FLAC_TAG) == FLAC_TAG)
				meta = metaVorbis(filename, tag_type);
	/*
	 * Getting ID3v2 metadata
	 * This is f***ing nasty code...
	 *
	 */
	else if((tag_type[1] & ID3V2_TAG) == ID3V2_TAG) meta = metaID3v2(filename);
	/* Getting ID3v1 metadata.  This is easy. */
	else if((tag_type[1] & ID3V1_TAG) == ID3V1_TAG) meta = metaID3v1(filename);

	return meta;
}

void free_meta(metadata *meta)
{
	free(meta->artist);
	free(meta->title);
	free(meta->mb);
	free(meta->album);
}

void clean_meta(metadata *meta)
{
	meta->artist = NULL;
	meta->title = NULL;
	meta->mb = NULL;
	meta->album = NULL;
	meta->tag_type[0] = NO_TAGS;
	meta->tag_type[1] = NO_TAGS;
}
