#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <curl/curl.h>
#include <stdio.h>
#include "fmt.h"
#include "md5.h"
#include "tags.h"
#include "queue.h"
#include "scrobbler.h"
#include "config.h"

#define SCROBBLER_HS_URL "http://post.audioscrobbler.com"
#define SCROBBLER_CLI_ID "xms"
#define SCROBBLER_HS_WAIT 1800
#define SCROBBLER_VERSION "1.1"

/* Scrobblerbackend for xmms plugin, first draft */

static int	sc_hs_status,
		sc_hs_timeout,
		sc_errors,
		sc_submit_interval,
		sc_submit_timeout,
		sc_srv_res_size,
		sc_giveup;

static char 	*sc_submit_url,
		*sc_username,
		*sc_password,
		*sc_challenge_hash,
		*sc_response_hash,
		*sc_srv_res,
		sc_curl_errbuf[CURL_ERROR_SIZE];


static size_t sc_store_res(void *ptr, size_t size,
		size_t nmemb,
		void *stream)
{
	int len = size * nmemb;

	if(sc_srv_res == NULL)
		sc_srv_res = realloc(sc_srv_res,
			sc_srv_res_size + len);
	else
		realloc(sc_srv_res, sc_srv_res_size + len);
	memcpy(sc_srv_res + sc_srv_res_size,
			ptr, len);
	sc_srv_res_size += len;
	return len;
}

static void sc_free_res(void)
{
	free(sc_srv_res);
	sc_srv_res = NULL;
	sc_srv_res_size = 0;
}

static int sc_parse_hs_res(void)
{
	char *interval;

	if (!sc_srv_res_size) {
		pdebug("No reply from server");
		return -1;
	}
	*(sc_srv_res + sc_srv_res_size) = 0;

	if (!strncmp(sc_srv_res, "FAILED ", 7)) {
		interval = strstr(sc_srv_res, "INTERVAL");
		if(!interval) {
			pdebug("missing INTERVAL");
		}
		else
		{
			*(interval - 1) = 0;
			sc_submit_interval = strtol(interval + 8, NULL, 10);
		}
		pdebug(fmt_vastr("error: %s", sc_srv_res));
		return -1;
	}
	if (!strncmp(sc_srv_res, "UPDATE ", 7)) {
		interval = strstr(sc_srv_res, "INTERVAL");
		if(!interval)
			pdebug("missing INTERVAL");
		else
		{
			*(interval - 1) = 0;
			sc_submit_interval = strtol(interval + 8, NULL, 10);
		}
		sc_submit_url = strchr(strchr(sc_srv_res, '\n') + 1, '\n') + 1;
		*(sc_submit_url - 1) = 0;
		sc_submit_url = strdup(sc_submit_url);
		sc_challenge_hash = strchr(sc_srv_res, '\n') + 1;
		*(sc_challenge_hash - 1) = 0;
		sc_challenge_hash = strdup(sc_challenge_hash);
		pdebug(fmt_vastr("update client: %s", sc_srv_res + 8));
		/* Russ isn't clear on whether we can submit with a not-updated
		 * client.  Neither is RJ.  I assume we can.
		 *
		 * sc_giveup = -1;
		 * return -1;
		 */
		return 0;
	}
	if (!strncmp(sc_srv_res, "UPTODATE\n", 9)) {
		interval = strstr(sc_srv_res, "INTERVAL");
		if (!interval) {
			pdebug("missing INTERVAL");
			/*
			 * This is probably a bad thing, but Russ seems to
			 * think its OK to assume that an UPTODATE response
			 * may not have an INTERVAL...  But we assume a default
			 * interval of 100, so this code should be safe.
			 *
			 * return -1;
			 */
		}
		else
		{
			*(interval - 1) = 0;
			sc_submit_interval = strtol(interval + 8, NULL, 10);
		}
		sc_submit_url = strchr(strchr(sc_srv_res, '\n') + 1, '\n') + 1;
		*(sc_submit_url - 1) = 0;
		sc_submit_url = strdup(sc_submit_url);
		sc_challenge_hash = strchr(sc_srv_res, '\n') + 1;
		*(sc_challenge_hash - 1) = 0;
		sc_challenge_hash = strdup(sc_challenge_hash);
		return 0;
	}
	if(!strncmp(sc_srv_res, "BADUSER\n", 8)) {
		interval = strstr(sc_srv_res, "INTERVAL");
		if(!interval)
			pdebug("missing INTERVAL");
		else
		{
			*(interval - 1) = 0;
			sc_submit_interval = strtol(interval + 8, NULL, 10);
		}
		return -1;
	}
	pdebug(fmt_vastr("unknown server-reply '%s'", sc_srv_res));
	return -1;
}

static char *hexify(char *pass, int len)
{
	static char buf[33], *bp = buf;
	char hexchars[] = "0123456789abcdef";
	int i;
	
	memset(buf, 0, sizeof(buf));
	
	for(i = 0; i < len; i++) {
		*(bp++) = hexchars[(pass[i] >> 4) & 0x0f];
		*(bp++) = hexchars[pass[i] & 0x0f];
	}
	*bp = 0;
	return buf;
}

static int sc_handshake(void)
{
	int status;
	char buf[4096];
	CURL *curl;

	snprintf(buf, sizeof(buf), "%s/?hs=true&p=%s&c=%s&v=%s&u=%s",
			SCROBBLER_HS_URL, SCROBBLER_VERSION,
			SCROBBLER_CLI_ID, VERSION, sc_username);
	curl = curl_easy_init();
	curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1);
	curl_easy_setopt(curl, CURLOPT_URL, buf);
	curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, 
			sc_store_res);
	memset(sc_curl_errbuf, 0, sizeof(sc_curl_errbuf));
	curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, sc_curl_errbuf);
	status = curl_easy_perform(curl);
	curl_easy_cleanup(curl);
	sc_hs_timeout = time(NULL) + SCROBBLER_HS_WAIT;
	if (status) {
		pdebug(sc_curl_errbuf);
		sc_errors++;
		sc_free_res();
		return -1;
	} 
	if (sc_parse_hs_res()) {
		sc_errors++;
		sc_free_res();
		return -1;
	}
	/*if (sc_challenge_hash != NULL) {
		md5_state_t md5state;
		unsigned char md5pword[16];
		
		md5_init(&md5state);
		pdebug(sc_challenge_hash);
		md5_append(&md5state, (unsigned const char *)sc_challenge_hash,
				strlen(sc_challenge_hash));
		pdebug(sc_password);
		md5_append(&md5state, (unsigned const char *)sc_password,
				strlen(sc_password));
		md5_finish(&md5state, md5pword);
		sc_response_hash = hexify(md5pword,
				sizeof(md5pword));
		pdebug(sc_response_hash);
	}*/
	sc_errors = 0;
	sc_hs_status = 1;
	sc_free_res();
	pdebug(fmt_vastr("submiturl: %s - interval: %d", 
				sc_submit_url, sc_submit_interval));
	return 0;
}

static int sc_parse_sb_res(void)
{
	char *ch;
#ifdef ALLOW_MULTIPLE
	int i;
#endif

	if (!sc_srv_res_size) {
		pdebug("No response from server");
		return -1;
	}
	*(sc_srv_res + sc_srv_res_size) = 0;
	if (!strncmp(sc_srv_res, "OK", 2)) {
#ifdef ALLOW_MULTIPLE
		q_free();
#else
		q_pop();
#endif
		if ((ch = strstr(sc_srv_res, "INTERVAL"))) {
			sc_submit_interval = strtol(ch + 8, NULL, 10);
			pdebug(fmt_vastr("got new interval: %d",
						sc_submit_interval));
		}
		pdebug(fmt_vastr("submission ok: %s", sc_srv_res));
		return 0;
	}
	if (!strncmp(sc_srv_res, "BADAUTH", 7)) {
		if ((ch = strstr(sc_srv_res, "INTERVAL"))) {
			sc_submit_interval = strtol(ch + 8, NULL, 10);
			pdebug(fmt_vastr("got new interval: %d",
						sc_submit_interval));
		}
		pdebug("incorrect username/password");
		sc_giveup = 1;
		return -1;
	}
	if (!strncmp(sc_srv_res, "FAILED", 6))  {
		if ((ch = strstr(sc_srv_res, "INTERVAL"))) {
			sc_submit_interval = strtol(ch + 8, NULL, 10);
			pdebug(fmt_vastr("got new interval: %d",
						sc_submit_interval));
		}
		pdebug(sc_srv_res);
		sc_errors++;
		return -1;
	}
	pdebug(fmt_vastr("unknown server-reply %s", sc_srv_res));
	return -1;
}

static char *sc_itemtag(char c, int n)
{
	static char buf[128];
	snprintf(buf, sizeof(buf), "%c[%d]", c, n);
	return buf;
}

#define cfa(f, l, n, v) \
curl_formadd(f, l, CURLFORM_COPYNAME, n, \
		CURLFORM_PTRCONTENTS, v, CURLFORM_END)
static int sc_submitentry(void)
{
	CURL *curl;
	struct HttpPost *post = NULL, *last = NULL;
	int status, i;
	item_t *item;

	curl = curl_easy_init();
	curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1);
	curl_easy_setopt(curl, CURLOPT_URL, sc_submit_url);
	curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION,
			sc_store_res);
	curl_easy_setopt(curl, CURLOPT_USERAGENT, USER_AGENT);
	cfa(&post, &last, "u", sc_username);
	cfa(&post, &last, "s", sc_response_hash);
	i = 0;
#ifdef ALLOW_MULTIPLE
	q_peekall(1);
	while ((item = q_peekall(0))) {
#else
		item = q_peek();
#endif
		cfa(&post, &last, sc_itemtag('a', i), I_ARTIST(item));
		cfa(&post, &last, sc_itemtag('t', i), I_TITLE(item));
		cfa(&post, &last, sc_itemtag('l', i), I_LEN(item));
		cfa(&post, &last, sc_itemtag('i', i), I_TIME(item));
		cfa(&post, &last, sc_itemtag('m', i), I_MB(item));
		cfa(&post, &last, sc_itemtag('b', i), I_ALBUM(item));
		pdebug(fmt_vastr("a[%d]=%s t[%d]=%s l[%d]=%s i[%d]=%s m[%d]=%s b[%d]=%s",
				i, I_ARTIST(item),
				i, I_TITLE(item),
				i, I_LEN(item),
				i, I_TIME(item),
				i, I_MB(item),
				i, I_ALBUM(item)));
#ifdef ALLOW_MULTIPLE
		i++;
	}
#endif
	curl_easy_setopt(curl, CURLOPT_HTTPPOST, post);
	memset(sc_curl_errbuf, 0, sizeof(sc_curl_errbuf));
	curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, sc_curl_errbuf);
	status = curl_easy_perform(curl);
	curl_formfree(post);
	curl_easy_cleanup(curl);

	if (status) {
		pdebug(sc_curl_errbuf);
		sc_errors++;
		sc_free_res();
		return -1;
	}
	if (sc_parse_sb_res()) {
		sc_errors++;
		sc_free_res();
		pdebug(fmt_vastr("Retrying in %d secs, %d elements in queue",
					sc_submit_interval, q_len()));
		return -1;
	}
	sc_free_res();
	return 0;
}

static void sc_handlequeue(void)
{
	if (!q_len())
		return;
	if (sc_submit_timeout < time(NULL)) {
		sc_submitentry();
		sc_submit_timeout = time(NULL) +
			sc_submit_interval;
	}
}

static void dump_queue(void)
{
	FILE *fd;
	item_t *item;
	char *home, buf[PATH_MAX];

	if (!(home = getenv("HOME")))
		return;

	snprintf(buf, sizeof(buf), "%s/.xmms/scrobblerqueue.txt", home);
	if (!(fd = fopen(buf, "w")))
		return;
	printf("Opening %s\n", buf);
	q_peekall(1);
	while ((item = q_peekall(0))) {
		fprintf(fd, "%s %s %s %s %s %s\n",
					I_ARTIST(item),
					I_TITLE(item),
					I_LEN(item),
					I_TIME(item),
					I_ALBUM(item),
					I_MB(item));
	}
	fclose(fd);
}

static void sc_cleaner(void)
{
	free(sc_submit_url);
	free(sc_username);
	free(sc_password);
	free(sc_challenge_hash);
	free(sc_srv_res);
	dump_queue();
	q_free();
	pdebug("scrobbler shutting down");
}

static void sc_checkhandshake(void)
{
	if (sc_errors > 2)
		sc_hs_status = 0;
	if (sc_hs_status)
		return;
	if (sc_errors > 2) {
		if (sc_hs_timeout < time(NULL))
			sc_handshake();
		return;
	}
	sc_handshake();
}

/**** Public *****/

/* Called at session startup*/

void sc_init(char *uname, char *pwd)
{
	sc_hs_status = sc_hs_timeout = sc_errors = sc_submit_timeout =
		sc_srv_res_size = sc_giveup = 0;
	sc_submit_interval = 100;

	sc_submit_url = sc_username = sc_password = sc_srv_res =
		sc_challenge_hash = NULL;
	sc_username = strdup(uname);
	sc_password = strdup(pwd);
	atexit(sc_cleaner);
	pdebug("scrobbler starting up");
}

void sc_addentry(metadata meta, int len)
{
	q_push(meta, len);
}

/* Call periodically from the plugin */
int sc_idle(void)
{
	sc_checkhandshake();
	if (sc_hs_status)
		sc_handlequeue();
	return sc_giveup;
}
