#include <xmms/plugin.h>
#include <xmms/configfile.h>
#include <xmms/xmmsctrl.h>
#include <pthread.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <wchar.h>
#include "tags.h"
#include "scrobbler.h"
#include "gtkstuff.h"
#include "config.h"
#include "fmt.h"
#include "unicode.h"

static void init(void);
static void cleanup(void);
static void *xs_thread(void *);
static int going;

static pthread_t pt_scrobbler;
static pthread_mutex_t m_scrobbler = PTHREAD_MUTEX_INITIALIZER;

static GeneralPlugin xmms_scrobbler =
{
	NULL,
	NULL,
	-1,
	NULL,
	init,
	about_show,
	configure_dialog,
	cleanup
};

static void init(void)
{
	char *username = NULL, *password = NULL;
	ConfigFile *cfgfile;
	going = 1;

	if ((cfgfile = xmms_cfg_open_default_file()) != NULL) {
		xmms_cfg_read_string(cfgfile, "audioscrobbler", "username",
				&username);
		xmms_cfg_read_string(cfgfile, "audioscrobbler", "password",
				&password);
		xmms_cfg_free(cfgfile);
	}
	if ((!username || !password) || (!*username || !*password)) {
		pdebug("username/password not found - not starting");
		going = 0;
		return;
	}
	sc_init(username, password);
	g_free(username);
	g_free(password);
	if (pthread_create(&pt_scrobbler, NULL, xs_thread, NULL)) {
		pdebug("Error creating thread");
		going = 0;
		return;
	}
	pdebug("plugin started");
}

static void cleanup(void)
{
	void *dummy;

	if (!going)
		return;
	pdebug("about to lock mutex");
	pthread_mutex_lock(&m_scrobbler);
	pdebug("locked mutex");
	going = 0;
	pthread_mutex_unlock(&m_scrobbler);
	pdebug("joining thread");
	pthread_join(pt_scrobbler, &dummy);
}

static char ishttp(const char *a)
{
	char *tmp, *bp;
	int status = 0;

	if (!a || !*a)
		return 0;

	tmp = strdup(a);
	for (bp = tmp; *bp; bp++)
		*bp = toupper(*bp);
	if (strstr(tmp, "HTTP://"))
		status = -1;
	free(tmp);
	return status;
}

static void *xs_thread(void *data)
{
	int playing, pos, len, prevpos = -1, prevlen = -1,
		otime = 0, oldtime = 0, dosubmit, run = 1;
	
	while (run) {
		playing = xmms_remote_is_playing(0);
		if (!playing) {
			prevpos = -1;
			prevlen = -1;
		}
		pos = xmms_remote_get_playlist_pos(0);
		len = xmms_remote_get_playlist_time(0, pos);
		oldtime = otime;
		otime = xmms_remote_get_output_time(0);
		dosubmit = (otime/1000 > 120) ||
			(((double)otime/((double)len + 1) * 100) >= 50);
		if(otime - oldtime > 1000 && (prevpos != pos && prevlen != len))
		{
			pdebug("No skipping allowed.  Discarding song.");
			prevpos = pos;
			prevlen = len;
		}
		if((pos != prevpos && len != prevlen) && playing && dosubmit) {
			char *fname, /**title, *artist,*/ *tmp = NULL, *sep, *tags;
			metadata_t *meta;

			prevpos = pos;
			prevlen = len;
			
			meta = malloc(sizeof(metadata_t));

			clean_meta(meta);

			fname = xmms_remote_get_playlist_file(0, pos);
			if (ishttp(fname)) {
				g_free(fname);
				continue;
			}
			tags = tag_exists(fname);
			if(tags[0] == CD_AUDIO)
			{
				ConfigFile *cfgfile;

				if ((cfgfile = xmms_cfg_open_default_file())
					!= NULL)
				{
					tmp = strrchr(fname, '.');
					*tmp = '\0';
					tags[1] = (char)atoi(tmp - 2);
					pdebug(fmt_vastr("Track: %d", tags[1]));
					xmms_cfg_read_string(cfgfile, "CDDA",
						"device", &fname);
					xmms_cfg_free(cfgfile);
					pdebug(fmt_vastr("CD Device: %s",
						fname));
				}
			}
			if((tags[0] != NO_TAGS && (tags[0] & READ_ERROR)
				!= READ_ERROR) || tags[1] != NO_TAGS)
			{
				free_meta(meta);
				meta = get_tag_data(fname, tags);
			}
			/*
			 * Fallback on the parsing mechanism (for now) if there
			 * are no tags.
			 */
			else
			{
				tmp = xmms_remote_get_playlist_title(0, pos);
				if ((sep = strstr(tmp, " - "))) {
					/* Look - this is not so bad is it? */
					*sep = 0;
					*(sep + 1) = 0;
					meta->artist = malloc(strlen(tmp) + 1);
					meta->title = malloc(strlen(sep + 3) +
						1);
					memset(meta->artist, '\0', strlen(tmp) +
						1);
					memset(meta->title, '\0',
						strlen(sep + 3) + 1);
					strcpy(meta->artist, tmp);
					strcpy(meta->title, sep + 3);
				}
			}
			if(meta->artist != NULL && meta->title != NULL)
			{
				pdebug(fmt_vastr(
					"submitting artist: %s, title: %s",
					meta->artist, meta->title));
				sc_addentry(meta, len/1000);
			}
			else
				pdebug("couldn't determine artist - "
						"title, not submitting");
			g_free(tmp);
			g_free(fname);
			free_meta(meta);
		}
		pthread_mutex_lock(&m_scrobbler);
		run = going;
		pthread_mutex_unlock(&m_scrobbler);
		if (sc_idle()) {
			pdebug("Giving up due to fatal error");
			run = 0;
		}
		usleep(100000);
	}
	pdebug("thread: exiting");
	pthread_exit(NULL);
}

GeneralPlugin *get_gplugin_info(void)
{
	xmms_scrobbler.description = g_strdup_printf("xmms scrobbler %s",
							VERSION);
	return &xmms_scrobbler;
}
