#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tags/id3v2.h"
#include "tags/id3v1.h"
#include "tags/vorbis.h"
#include "tags/itunes.h"
#include "tags/ape.h"
#include "tags/cdaudio.h"
#include "fmt.h"
#include "tags.h"
#include "unicode.h"

char *tag_exists(char *filename)
{
	FILE *fp;
	int status = 0;
	static char tags[2];

	tags[0] = NO_TAGS;
	tags[1] = NO_TAGS;

	/* Check for CD Audio */
	if(findCDAudio(filename))
	{
		pdebug("Found CD Audio...");
		tags[0] = CD_AUDIO;

		return tags;
	}

	fp = fopen(filename, "r");
	if(fp == NULL)
	{
		pdebug("Error reading file...");
		tags[0] = READ_ERROR;

		return tags;
	}

	/* Check for ID3v1 */
	fseek(fp, -128, SEEK_END);
	if(findID3v1(fp))
	{
		pdebug("Found ID3v1 tag...");
		tags[1] |= ID3V1_TAG;
	}

	/* Check for ID3v2 */
	fseek(fp, 0, SEEK_SET);
	status = findID3v2(fp);
	if(status > -1)
	{
		pdebug("Found ID3v2 tag...");
		tags[1] |= ID3V2_TAG;
	}
	status = 0;

	/* Check for Ogg Vorbis */
	fseek(fp, 0, SEEK_SET);
	status = findVorbis(fp);
	if(status > -1)
	{
		pdebug("Found Vorbis comment block...");
		tags[1] |= VORBIS_TAG;
	}
	status = 0;

	/* Check for FLAC */
	fseek(fp, 0, SEEK_SET);
	if(findFlac(fp))
	{
		pdebug("Found FLAC tag...");
		tags[1] |= FLAC_TAG;
	}
	
	/* Check for OggFLAC */
	fseek(fp, 0, SEEK_SET);
	status = findOggFlac(fp);
	if(status > -1)
	{
		pdebug("Found OggFLAC...");
		tags[1] |= OGGFLAC_TAG;
	}
	status = 0;

	/* Check for Speex */
	fseek(fp, 0, SEEK_SET);
	status = findSpeex(fp);
	if(status > -1)
	{
		pdebug("Found Speex...");
		tags[1] |= SPEEX_TAG;
	}
	status = 0;

	/* Check for APE */
	fseek(fp, 0, SEEK_SET);
	status = findAPE(fp);
	if(status > 0)
	{
		pdebug("Found APE tag...");
		tags[1] |= APE_TAG;
	}
	status = 0;
	
	/* Check for iTunes M4A */
	fseek(fp, 0, SEEK_SET);
	status = findiTunes(fp);
	if(status > -1)
	{
		pdebug("Found iTunes tag...");
		tags[1] |= ITUNES_TAG;
	}

	fclose(fp);

	return tags;
}

static metadata_t *metaCD(char *filename, char track)
{
	metadata_t *meta;
	cdaudio_t *musicbrainz;
	
	meta = malloc(sizeof(metadata_t));

	clean_meta(meta);

	pdebug("Getting CD Audio metadata...");
	musicbrainz = readCDAudio(filename, track);
	if(musicbrainz == NULL)
	{
		pdebug("Error getting metadata");
		return meta;
	}
	pdebug("Reading metadata into structs...");
	meta->album = malloc(strlen(musicbrainz->album) + 1);
	strcpy(meta->album, musicbrainz->album);
	meta->artist = malloc(strlen(musicbrainz->artist) + 1);
	strcpy(meta->artist, musicbrainz->artist);
	meta->title = malloc(strlen(musicbrainz->title) + 1);
	strcpy(meta->title, musicbrainz->title);
	meta->mb = malloc(strlen(musicbrainz->mbid) + 1);
	strcpy(meta->mb, musicbrainz->mbid);
	
	freeCDAudio(musicbrainz);

	return meta;
} /* End CD Audio support */

static metadata_t *metaAPE(char *filename)
{
	metadata_t *meta;
	ape_t *ape;
	int i, t = 0;
	
	meta = malloc(sizeof(metadata_t));
	
	clean_meta(meta);

	pdebug("Getting APE tag metadata...");
	ape = readAPE(filename);

	for(i = 0; i < ape->numitems; i++)
	{
		apefielddata_t *item = ape->items[i];
		
		if(!strcmp(item->name, "Title"))
		{
			pdebug("Found Title!");
			meta->title = realloc(meta->title, item->len + 1);
			strcpy(meta->title, item->data);
		}
		else if(!strcmp(item->name, "Artist"))
		{
			pdebug("Found Artist!");
			meta->artist = realloc(meta->artist, item->len + 1);
			strcpy(meta->artist, item->data);
		}
		else if(strcmp(item->name, "Album") == 0)
		{
			pdebug("Found Album!");
			meta->album = realloc(meta->album, item->len + 1);
			strcpy(meta->album, item->data);
		}
		else if(strcmp(item->name, "Comment") == 0)
		{
			unsigned char *comment_value = NULL, *eq_pos, *sep_pos,
					*subvalue;

			subvalue = item->data;
			sep_pos = strchr(subvalue, '|');
			while(sep_pos != NULL || t == 0)
			{
				t = 1;
				if(sep_pos != NULL)
					*sep_pos = '\0';
				comment_value = realloc(comment_value,
							strlen(subvalue) + 1);
				strcpy(comment_value, subvalue);
				if(sep_pos != NULL)
					sep_pos++;
				eq_pos = strchr(comment_value, '=');
				if(eq_pos != NULL)
				{
					*eq_pos = '\0';
					eq_pos++;
					if(!strcmp(comment_value,
						"musicbrainz_trackid"))
					{
						/* ??? */
						pdebug("Found MusicBrainz Track ID!");
						meta->mb = realloc(meta->mb,
							strlen(eq_pos) + 1);
						strcpy(meta->mb, eq_pos);
						break;
					}
				}
				if(sep_pos != NULL)
				{
					t = 0;
					subvalue = sep_pos;
					sep_pos = strchr(subvalue, '|');
				}
			}
			t = 0;
			if(comment_value != NULL)
				free(comment_value);
		}
	}

	freeAPE(ape);

	meta->tag_type[1] = APE_TAG;

	return meta;
} /* End APE support */

static metadata_t *metaVorbis(char *filename, char tag_type[2])
{
	metadata_t *meta;
	vorbis_t *comments = NULL;
	int i;
	
	meta = malloc(sizeof(metadata_t));
	
	clean_meta(meta);

	/* Several slightly different methods of getting the data... */
	if((tag_type[1] & VORBIS_TAG) == VORBIS_TAG)
	{
		pdebug("Getting Vorbis comment metadata...");
		comments = readVorbis(filename);
		meta->tag_type[1] = VORBIS_TAG;
	}
	else if((tag_type[1] & FLAC_TAG) == FLAC_TAG)
	{
		pdebug("Getting FLAC tag metadata...");
		comments = readFlac(filename);
		meta->tag_type[1] = FLAC_TAG;
	}
	else if((tag_type[1] & OGGFLAC_TAG) == OGGFLAC_TAG)
	{
		pdebug("Getting OggFLAC tag metadata...");
		comments = readOggFlac(filename);
		meta->tag_type[1] = OGGFLAC_TAG;
	}
	else if((tag_type[1] & SPEEX_TAG) == SPEEX_TAG)
	{
		pdebug("Getting Speex tag metadata...");
		comments = readSpeex(filename);
		meta->tag_type[1] = SPEEX_TAG;
	}
	if(comments == NULL)
	{
		pdebug("Error in Vorbis Comment read");
		return meta;
	}
	for(i = 0; i < comments->numitems; i++)
	{
		vorbisfielddata_t *field = comments->items[i];
		if(!fmt_strcasecmp(field->name, "TITLE"))
		{
			pdebug("Found Title!");
			
			meta->title = realloc(meta->title, strlen(field->data)
						+ 1);
			strcpy(meta->title, field->data);
		}
		else if(!fmt_strcasecmp(field->name, "PERFORMER"))
		{
			pdebug("Found Artist!");

			meta->artist = realloc(meta->artist, strlen(field->data)
						+ 1);
			strcpy(meta->artist, field->data);
		}
		else if(!fmt_strcasecmp(field->name, "ARTIST") && meta->artist == NULL)
		{
			pdebug("Found Artist!");

			meta->artist = realloc(meta->artist, strlen(field->data)
						+ 1);
			strcpy(meta->artist, field->data);
		}
		else if(!fmt_strcasecmp(field->name, "ALBUM"))
		{
			pdebug("Found Album!");
			
			meta->album = realloc(meta->album, strlen(field->data)
						+ 1);
			strcpy(meta->album, field->data);
		}
		else if(!fmt_strcasecmp(field->name, "MUSICBRAINZ_TRACKID"))
		{
			pdebug("Found MusicBrainz Track ID!");
			
			meta->mb = realloc(meta->mb, strlen(field->data) + 1);
			strcpy(meta->mb, field->data);
		}
	}
	
	freeVorbis(comments);
	
	return meta;
} /* End Vorbis/FLAC/OggFLAC/Speex support */

static metadata_t *metaID3v2(char *filename)
{
	metadata_t *meta;
	id3v2_t *id3v2;
	int i;
	
	meta = malloc(sizeof(metadata_t));

	clean_meta(meta);

	pdebug("Getting ID3v2 tag metadata...");
	id3v2 = readID3v2(filename);
	if(id3v2 == NULL)
	{
		pdebug("Error in readID3v2()");
		return meta;
	}
	for(i = 0; i < id3v2->numitems; i++)
	{
		unsigned char *data = NULL, *utf = NULL;
		framedata_t *frame = id3v2->items[i];
		if(	(id3v2->version == 2 && frame->frameid == ID3V22_TT2) ||
			(id3v2->version == 3 && frame->frameid == ID3V23_TIT2) ||
			(id3v2->version == 4 && frame->frameid == ID3V24_TIT2))
		{
			pdebug("Found Title!");
			
			utf = ID3v2_parseText(frame);

			meta->title = realloc(meta->title, strlen(utf) + 1);
			strcpy(meta->title, utf);
		}
		else if((id3v2->version == 2 && frame->frameid == ID3V22_TP1) ||
			(id3v2->version == 3 && frame->frameid == ID3V23_TPE1) ||
			(id3v2->version == 4 && frame->frameid == ID3V24_TPE1))
		{
			pdebug("Found Artist!");

			utf = ID3v2_parseText(frame);

			meta->artist = realloc(meta->artist, strlen(utf) + 1);
			strcpy(meta->artist, utf);
		}
		else if((id3v2->version == 2 && frame->frameid == ID3V22_TAL) ||
			(id3v2->version == 3 && frame->frameid == ID3V23_TALB) ||
			(id3v2->version == 4 && frame->frameid == ID3V24_TALB))
		{
			pdebug("Found Album!");
			
			utf = ID3v2_parseText(frame);
			
			meta->album = realloc(meta->album, strlen(utf) + 1);
			strcpy(meta->album, utf);
		}
		else if((id3v2->version == 2 && frame->frameid == ID3V22_UFI) ||
			(id3v2->version == 3 && frame->frameid == ID3V23_UFID) ||
			(id3v2->version == 4 && frame->frameid == ID3V24_UFID))
		{
			data = ID3v2_getData(frame);
			
			if(!strcmp(data, "http://musicbrainz.org"))
			{
				pdebug("Found MusicBrainz Track ID!");
				
				utf = data + 23;

				meta->mb = realloc(meta->mb, strlen(utf) + 1);
				strcpy(meta->mb, utf);
			}
		}
		
		if(data != NULL)
			free(data);
		else if(utf != NULL)
			free(utf);
	}
	
	freeID3v2(id3v2);

	meta->tag_type[1] = ID3V2_TAG;

	return meta;
} /* End ID3v2 support */

static metadata_t *metaiTunes(char *filename)
{
	metadata_t *meta;
	itunes_t *itunes;
	
	meta = malloc(sizeof(metadata_t));
	
	clean_meta(meta);
	
	pdebug("Getting iTunes tag metadata...");	
	itunes = readiTunes(filename);
	if(itunes->title != NULL)
	{
		pdebug("Found Title!");
			
		meta->title = realloc(meta->title, strlen(itunes->title) + 1);
		strcpy(meta->title, itunes->title);
	}
	if(itunes->artist != NULL)
	{
		pdebug("Found Artist!");
			
		meta->artist = realloc(meta->artist, strlen(itunes->artist) + 1);
		strcpy(meta->artist, itunes->artist);
	}
	if(itunes->album != NULL)
	{
		pdebug("Found Album!");
			
		meta->album = realloc(meta->album, strlen(itunes->album) + 1);
		strcpy(meta->album, itunes->album);
	}
	
	freeiTunes(itunes);
	
	meta->tag_type[1] = ITUNES_TAG;

	return meta;
} /* End M4A support */

static metadata_t *metaID3v1(char *filename)
{
	metadata_t *meta;
	id3v1_t *id3v1;
	
	meta = malloc(sizeof(metadata_t));
	
	clean_meta(meta);
	
	pdebug("Getting ID3v1 tag metadata...");	
	id3v1 = readID3v1(filename);
	if(strcmp(id3v1->title, ""))
	{
		pdebug("Found Title!");
			
		meta->title = realloc(meta->title, strlen(id3v1->title) + 1);
		strcpy(meta->title, id3v1->title);
	}
	if(strcmp(id3v1->artist, ""))
	{
		pdebug("Found Artist!");
			
		meta->artist = realloc(meta->artist, strlen(id3v1->artist) + 1);
		strcpy(meta->artist, id3v1->artist);
	}
	if(strcmp(id3v1->album, ""))
	{
		pdebug("Found Album!");
			
		meta->album = realloc(meta->album, strlen(id3v1->album) + 1);
		strcpy(meta->album, id3v1->album);
	}
	/*
	 * This next one is unofficial, but maybe someone will use it.
	 * I don't think anyone's going to trigger this by accident.
	 *
	 * Specification:
	 * In comment field (ID3v1 or ID3v1.1):
	 * 1 byte: \0
	 * 10 bytes: "MBTRACKID=" identifier
	 * 16 bytes: binary representation of Track ID
	 * 1 byte: \0
	 *
	 * e.g. for "Missing" on Evanescence's "Bring Me to Life" single
	 *
	 * Track ID = 9c2567cb-4a8b-4096-b105-dada6b95a08b
	 *
	 * Therefore, comment field would equal:
	 * "MBID: \156%gJ\139@\150\005k\149\160\139"
	 * in ISO 8859-1 (with three digit escape characters)
	 */
	if(!strncmp((id3v1->comment) + 1, "MBTRACKID=", 10))
	{
		pdebug("Found MusicBrainz Track ID!");
		
		meta->mb = realloc(meta->mb, 37);
		memset(meta->mb, '\0', 37);
		sprintf(meta->mb,
	"%02x%02x%02x%02x-%02x%02x-%02x%02x-%02x%02x-%02x%02x%02x%02x%02x%02x",
			id3v1->comment[11],
			id3v1->comment[12],
			id3v1->comment[13],
			id3v1->comment[14],
			id3v1->comment[15],
			id3v1->comment[16],
			id3v1->comment[17],
			id3v1->comment[18],
			id3v1->comment[19],
			id3v1->comment[20],
			id3v1->comment[21],
			id3v1->comment[22],
			id3v1->comment[23],
			id3v1->comment[24],
			id3v1->comment[25],
			id3v1->comment[26]);
	}
	
	freeID3v1(id3v1);
	
	meta->tag_type[1] = ID3V1_TAG;

	return meta;
} /* End ID3v1 support */

metadata_t *get_tag_data(char *filename, char tag_type[2])
{
	metadata_t *meta = NULL;

	/*
	 * Getting CD Audio metadata from MusicBrainz...
	 *
	 */
	if((tag_type[0] & CD_AUDIO) == CD_AUDIO)
		meta = metaCD(filename, tag_type[1]);
	/* Getting APE metadata */
	else if((tag_type[1] & APE_TAG) == APE_TAG)
		meta = metaAPE(filename);
	/* Getting Vorbis, FLAC, OggFLAC, and Speex metadata */
	else if((tag_type[1] & VORBIS_TAG) == VORBIS_TAG ||
		(tag_type[1] & FLAC_TAG) == FLAC_TAG ||
		(tag_type[1] & OGGFLAC_TAG) == OGGFLAC_TAG ||
		(tag_type[1] & SPEEX_TAG) == SPEEX_TAG)
			meta = metaVorbis(filename, tag_type);
	else if((tag_type[1] & ITUNES_TAG) == ITUNES_TAG)
		meta = metaiTunes(filename);
	/* Getting ID3v2 metadata */
	else if((tag_type[1] & ID3V2_TAG) == ID3V2_TAG)
		meta = metaID3v2(filename);
	/* Getting ID3v1 metadata.  This is easy. */
	else if((tag_type[1] & ID3V1_TAG) == ID3V1_TAG)
		meta = metaID3v1(filename);
	else
	{
		meta = malloc(sizeof(metadata_t));
		clean_meta(meta);
	}

	return meta;
}

void free_meta(metadata_t *meta)
{
	if(meta->artist != NULL)
		free(meta->artist);
	if(meta->title != NULL)
		free(meta->title);
	if(meta->mb != NULL)
		free(meta->mb);
	if(meta->album != NULL)
		free(meta->album);
	free(meta);
}

void clean_meta(metadata_t *meta)
{
	meta->artist = NULL;
	meta->title = NULL;
	meta->mb = NULL;
	meta->album = NULL;
	meta->tag_type[0] = NO_TAGS;
	meta->tag_type[1] = NO_TAGS;
}
